#ifndef ARRAY3D_H
#define ARRAY3D_H

#include <vector>
#include "Dim3D.h"
#include "Field3DImpl.h"
#include <iostream>

namespace CompuCell3D {

template <typename T>
class Array3D{
   public:
      typedef std::vector<std::vector<std::vector<T> > > ContainerType;
      void allocateArray(const Dim3D & _dim , T & val=T());
//       operator Type&();
      ContainerType & getContainer(){return array;}
      
   private:
     std::vector<std::vector<std::vector<T> > >  array;

};

template <typename T>
void Array3D<T>::allocateArray(const Dim3D & _dim , T & val){
   using namespace std;
//    vector<vector<T> > vec;
//    vec.assign(_dim.x,vector<T>(_dim.y,val));
   array.assign(_dim.x, vector<vector< T > >(_dim.y,vector< T >(_dim.z,val)));
}

// template <typename T>
// Array3D<T>::operator Array3D<T>::Type&(){
//    return  array;
// }



// template <typename T>
// Array3D<T>::operator Array3D<T>::Type&(){
//    return  array;
// }


//Adapter is necessary to keep current API of the Field3D . These fields are registered in Simulator and 
template <typename T>
class Array3DField3DAdapter:public Field3DImpl<T>{
   public:
      Array3DField3DAdapter() : Field3DImpl<T>(Dim3D(1,1,1),T()) , array3DPtr(0) , containerPtr(0){};
      virtual ~Array3DField3DAdapter(){
         if(array3DPtr)
            delete array3DPtr;
            array3DPtr=0;
      }
//       void setArray3DPtr(Array3D<T> * _ptr){ array3DPtr = _ptr;}
      Array3D<T> * getArray3DPtr(){ return array3DPtr; }
      typename Array3D<T>::ContainerType &  getContainer(){ return array3DPtr->getContainer(); }
      virtual void setDim(const Dim3D theDim) {
         if(!array3DPtr){
            array3DPtr=new Array3D<T>();
            T t;
            t=T();

            array3DPtr->allocateArray( theDim ,t);
            containerPtr=&array3DPtr->getContainer();
            dim=theDim;
         }else{
            delete array3DPtr;
            array3DPtr=new Array3D<T>();
            T t;
            t=T();
            array3DPtr->allocateArray( theDim, t );
            containerPtr=&array3DPtr->getContainer();
            dim=theDim;
         }
      }
      virtual Dim3D getDim() const {return dim;};
      virtual bool isValid(const Point3D &pt) const {      
         return (0 <= pt.x && pt.x < dim.x &&
	      0 <= pt.y && pt.y < dim.y &&
	      0 <= pt.z && pt.z < dim.z);
      }
      virtual void set(const Point3D &pt, const T value) {
         if(array3DPtr){
            (*containerPtr)[pt.x][pt.y][pt.z]=value;
         }
      }
      virtual T get(const Point3D &pt) const {
         return (*containerPtr)[pt.x][pt.y][pt.z];

      };
      virtual T getByIndex(long _offset) const {
         return T();
      }
      virtual void setByIndex(long _offset, const T _value) {
         
      }
   protected:
      Array3D<T> * array3DPtr;
      Dim3D dim;
      typename Array3D<T>::ContainerType * containerPtr;
};


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Adapter is necessary to keep current API of the Field3D . 3D array is allocated as a single chunk of memory - therefore trhe name linear. this particular array acts as a proxy for Fortran Arrays
//Fortran style array  shifts all the elements by (1,1,1) vector and 
//has dimensions (m+1)x(n+1)x(q+1)  
//last dimension of the array is iterated in the innermost loop 

template <typename T>
class Array3DLinearFortranField3DAdapter:public Field3DImpl<T>{
   public:
	  typedef T precision_t;

      Array3DLinearFortranField3DAdapter() : Field3DImpl<T>(Dim3D(1,1,1),T()) , containerPtr(0){};

      Array3DLinearFortranField3DAdapter(Dim3D & _dim,T & _initVal):Field3DImpl<T>(Dim3D(1,1,1),T()){
         allocateMemory(_dim,_initVal);         
      }      
      
      virtual ~Array3DLinearFortranField3DAdapter(){}
      
      virtual void allocateMemory(const Dim3D theDim,T & _initVal) {         
         container.clear();
         internalDim.x=theDim.x+1;
         internalDim.y=theDim.y+1;
         internalDim.z=theDim.z+1;

         //internalDim.x=theDim.x;
         //internalDim.y=theDim.y;
         //internalDim.z=theDim.z;

         dim=theDim;
         
         container.assign((internalDim.x)*(internalDim.y)*(internalDim.z),_initVal); 
         containerPtr=&(container[0]);         
         
      }
      
	  std::vector<T> & getContainerRef(){return container;}
	  T * getContainerArrayPtr(){return containerPtr;}

      inline unsigned int index(int _x,int _y,int _z ) const{
        //return _x+1 + (_y+1+ (_z+1) * internalDim.y) * internalDim.x;
		//return _x+1 + (_y+1+ (_z+1) * (internalDim.y)) * (internalDim.x);
		//return _z + (_y+ _x * (internalDim.y)) * (internalDim.z);//start indexing from 0'th element but calculate index based on increased lattice dimmension

		return _x + (_y+ _z * internalDim.y) * internalDim.x;//start indexing from 0'th element but calculate index based on increased lattice dimmension
		
		//return _x + (_y+ (_z) * internalDim.y) * internalDim.x;
        
       }
       
       
      virtual Dim3D getDim() const {return dim;}

      virtual bool isValid(const Point3D &pt) const {      
         return (0 <= pt.x && pt.x < dim.x &&
	      0 <= pt.y && pt.y < dim.y &&
	      0 <= pt.z && pt.z < dim.z);
      }

      virtual void set(const Point3D &pt, const T value) {
            container[index(pt.x,pt.y,pt.z)]=value;            
         
      }

      virtual T get(const Point3D &pt) const {
         return container[index(pt.x,pt.y,pt.z)];            

      }

      void setQuick(const Point3D &pt, const T value) {
            container[index(pt.x,pt.y,pt.z)]=value;            
         
      }
      T getQuick(const Point3D &pt) const {
         return container[index(pt.x,pt.y,pt.z)];            

      }

      void setQuick(int _x,int _y,int _z , T value) {
            container[index(_x,_y,_z)]=value;            
         
      }
      T getQuick(int _x,int _y,int _z ) const {
         return container[index(_x,_y,_z)];            

      }
      
      
      virtual T getByIndex(long _offset) const {
         return T();
      }
      virtual void setByIndex(long _offset, const T _value) {
         
      }
   protected:      
      std::vector<T> container;
      Dim3D dim;
      Dim3D internalDim;
      T * containerPtr;
};


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Adapter is necessary to keep current API of the Field3D . 2D array is allocated as a single chunk of memory - therefore trhe name linear. this particular array acts as a proxy for Fortran Arrays
//Fortran style array  shifts all the elements by (1,1) vector and 
//actually it looks like indexing begins at 0,0 but we need to use increaze lattice size by 1 in index incalculation to obtain correct index value
//has dimensions (m+1)x(n+1)
//last dimension of the array is iterated in the innermost loop 


template <typename T>
class Array2DLinearFortranField3DAdapter:public Field3DImpl<T>{
   public:
	  typedef T precision_t;
      Array2DLinearFortranField3DAdapter() : Field3DImpl<T>(Dim3D(1,1,1),T()) , containerPtr(0){};
      
      Array2DLinearFortranField3DAdapter(Dim3D & _dim,T & _initVal):Field3DImpl<T>(Dim3D(1,1,1),T())
	  {
         allocateMemory(_dim,_initVal);         
      }       
      virtual ~Array2DLinearFortranField3DAdapter(){}
      
      virtual void allocateMemory(const Dim3D theDim,T & _initVal) {         
         container.clear();
         internalDim.x=theDim.x+1;
         internalDim.y=theDim.y+1;
         internalDim.z=1;
         dim=theDim;
         dim.z=1;
         
         container.assign((internalDim.x)*(internalDim.y),_initVal); 
         containerPtr=&(container[0]);         
         
      }
      
	  std::vector<T> & getContainerRef(){return container;}
	  T * getContainerArrayPtr(){return containerPtr;}

      inline unsigned int index(int _x,int _y ) const{
        //return _x+1 + (_y+1) * internalDim.x;
		return _x + (_y) * internalDim.x; //start indexing from 0'th element but calculate index based on increased lattice dimmension
        
       }
              
      virtual Dim3D getDim() const {return dim;};
      virtual bool isValid(const Point3D &pt) const {      
         return (0 <= pt.x && pt.x < dim.x &&
	      0 <= pt.y && pt.y < dim.y);
      }
      virtual void set(const Point3D &pt, const T value) {
            container[index(pt.x,pt.y)]=value;            
         
      }
      virtual T get(const Point3D &pt) const {
		  //return Array2DLinearFortranField3DAdapter<T>::container[Array2DLinearFortranField3DAdapter<T>::indexPt(pt)];  
		 //return (T)indexPt(pt);
		 //return container[0];
   //      using namespace std;
		 //if(pt.y==0&&pt.z==0){
			//cerr<<"get "<<pt<<" = "<<container[Array2DLinearFortranField3DAdapter<T>::index(pt.x,pt.y)]<<endl;
			//cerr<<" index="<<index(pt.x,pt.y)<<" internalDim.x="<<internalDim.x<<endl;
			//}
		 return container[Array2DLinearFortranField3DAdapter<T>::index(pt.x,pt.y)];  

      };

      void setQuick(const Point3D &pt, const T value) {
            container[index(pt.x,pt.y)]=value;            
         
      }
      virtual T getQuick(const Point3D &pt) const {
         return container[index(pt.x,pt.y)];            

      }

      void setQuick(int _x,int _y , T value) {
            container[index(_x,_y)]=value;            
         
      }
      virtual T getQuick(int _x,int _y ) const {
         return container[index(_x,_y)];            

      };
      
      
      virtual T getByIndex(long _offset) const {
         return T();
      }
      virtual void setByIndex(long _offset, const T _value) {
         
      }
   protected:      
      std::vector<T> container;
      Dim3D dim;
      Dim3D internalDim;
      T * containerPtr;
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template <typename T>
class Array3DBorders{
   public:
      typedef T *** ContainerType;
      Array3DBorders():array(0),borderWidth(1){
      }
      virtual ~Array3DBorders(){freeMemory();};
      
      void allocateArray(const Dim3D & _dim , T & val=T());
//       operator Type&();
      ContainerType & getContainer(){return array;}
      void setBorderWidth(unsigned int _borderWidth){borderWidth=_borderWidth;}
      unsigned int getBorderWidth(){return borderWidth;}
      bool switchContainersQuick(Array3DBorders<T> & _array3D);
      Dim3D getInternalDim(){return internalDim;}
   protected:
     ContainerType array;
     unsigned int borderWidth;
     Dim3D internalDim;
   
     void allocateMemory(const Dim3D & _dim, T & val=T());
     void freeMemory();

};



template <typename T>
void Array3DBorders<T>::allocateArray(const Dim3D & _dim , T & val){
   
   internalDim=_dim;
   freeMemory();
   allocateMemory(internalDim,val);

}

template <typename T>
void Array3DBorders<T>::allocateMemory(const Dim3D & _dim, T & val){

   array=new T** [_dim.x];
   for(int i = 0  ; i < _dim.x ; ++i){
      array[i]=new T* [_dim.y];
   }

   for(int i = 0  ; i < _dim.x ; ++i)
      for(int j = 0  ; j < _dim.y ; ++j){
         array[i][j]=new T [_dim.z];
   }

   for(int i = 0  ; i < _dim.x ; ++i)
      for(int j = 0  ; j < _dim.y ; ++j)
         for(int k = 0  ; k < _dim.z ; ++k){
            array[i][j][k]=val;
         }

}

#include <iostream>
template <typename T>
void Array3DBorders<T>::freeMemory(){
using namespace std;

   if(array){
      
      for(int i = 0  ; i < internalDim.x ; ++i)
         for(int j = 0  ; j < internalDim.y ; ++j){
      
            delete [] array[i][j];
            array[i][j]=0;
         }

      for(int i = 0  ; i < internalDim.x ; ++i){
         delete [] array[i];
         array[i]=0;
      }
      delete [] array;
      array=0;

   }


}


template <typename T>
bool  Array3DBorders<T>::switchContainersQuick(Array3DBorders<T> & _switchedArray){
   ContainerType tmpPtr;
   ContainerType & switchedArrayPtr=_switchedArray.getContainer();

   tmpPtr=array;
   array=switchedArrayPtr;
   switchedArrayPtr=tmpPtr;
   return true;
}

//Adapter is necessary to keep current API of the Field3D . These fields are registered in Simulator and 
template <typename T>
class Array3DBordersField3DAdapter:public Field3DImpl<T>, public Array3DBorders<T>{
   public:
      Array3DBordersField3DAdapter() : Field3DImpl<T>(Dim3D(1,1,1),T()), Array3DBorders<T>() {};
      Array3DBordersField3DAdapter(Dim3D & _dim,T & _initVal):Field3DImpl<T>(Dim3D(1,1,1),T()), Array3DBorders<T>(){
         allocateMemory(_dim,_initVal);
         Array3DBorders<T>::internalDim=_dim;
      }
      virtual ~Array3DBordersField3DAdapter(){
//          if(array3DPtr)
//             delete array3DPtr;
//             array3DPtr=0;
      };
//       void setArray3DPtr(Array3D<T> * _ptr){ array3DPtr = _ptr;}
//       Array3DBorders<T> * getArray3DBordersPtr(){ return array3DPtr; }
//       typename Array3DBorders<T>::ContainerType &  getContainer(){ return getContainer(); }
      virtual void setDim(const Dim3D theDim) {
         T t;
         
         allocateMemory(theDim,t);
         Array3DBorders<T>::internalDim=theDim;
//          if(!array3DPtr){
//             array3DPtr=new Array3DBorders<T>();
//             T t;
//             t=T();
// 
//             array3DPtr->allocateArray( theDim ,t);
//             containerPtr=&array3DPtr->getContainer();
//             dim=theDim;
//          
//          }else{
//             delete array3DPtr;
//             array3DPtr=new Array3DBorders<T>();
//             T t;
//             t=T();
//             array3DPtr->allocateArray( theDim, t );
//             containerPtr=&array3DPtr->getContainer();
//             dim=theDim;
//          }
      }
      virtual Dim3D getDim() const {

         return Dim3D(Array3DBorders<T>::internalDim.x-2*Array3DBorders<T>::borderWidth,
                     Array3DBorders<T>::internalDim.y-2*Array3DBorders<T>::borderWidth,
                     Array3DBorders<T>::internalDim.z-2*Array3DBorders<T>::borderWidth);
      };
      virtual bool isValid(const Point3D &pt) const {      
         return (0 <= pt.x && pt.x < Array3DBorders<T>::internalDim.x &&
	      0 <= pt.y && pt.y < Array3DBorders<T>::internalDim.y &&
	      0 <= pt.z && pt.z < Array3DBorders<T>::internalDim.z);
      }
      virtual void set(const Point3D &pt, const T value) {
         
//             unsigned int borderWidth=array3DPtr->getBorderWidth();
            Array3DBorders<T>::array
            [pt.x+Array3DBorders<T>::borderWidth]
            [pt.y+Array3DBorders<T>::borderWidth]
            [pt.z+Array3DBorders<T>::borderWidth]=value;
         
      }
      virtual T get(const Point3D &pt) const {
//          unsigned int borderWidth=array3DPtr->getBorderWidth();
         return Array3DBorders<T>::array
                  [pt.x+Array3DBorders<T>::borderWidth]
                  [pt.y+Array3DBorders<T>::borderWidth]
                  [pt.z+Array3DBorders<T>::borderWidth];
      }

      virtual T getByIndex(long _offset) const {
         return T();
      }
      virtual void setByIndex(long _offset, const T _value) {
         
      }
   protected:
//       Array3DBorders<T> * array3DPtr;
//       Dim3D dim;
//       typename Array3DBorders<T>::ContainerType * containerPtr;
//       unsigned int borderWidth;
};


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
template <typename T>
class Array2DBorders{
   public:
      typedef T ** ContainerType;
      Array2DBorders():array(0),borderWidth(1){
      }
      virtual ~Array2DBorders(){freeMemory();};
      
      void allocateArray(const Dim3D & _dim , T & val=T());
//       operator Type&();
      ContainerType & getContainer(){return array;}
      void setBorderWidth(unsigned int _borderWidth){borderWidth=_borderWidth;}
      unsigned int getBorderWidth(){return borderWidth;}
      bool switchContainersQuick(Array2DBorders<T> & _array2D);
      Dim3D getInternalDim(){return internalDim;}
   protected:
     ContainerType array;
     unsigned int borderWidth;
     Dim3D internalDim;
   
     void allocateMemory(const Dim3D & _dim, T & val=T());
     void freeMemory();

};



template <typename T>
void Array2DBorders<T>::allocateArray(const Dim3D & _dim , T & val){
   
   internalDim=_dim;
   freeMemory();
   allocateMemory(internalDim,val);

}

template <typename T>
void Array2DBorders<T>::allocateMemory(const Dim3D & _dim, T & val){

   array=new T* [_dim.x];
   for(int i = 0  ; i < _dim.x ; ++i){
      array[i]=new T [_dim.y];
   }

   for(int i = 0  ; i < _dim.x ; ++i)
      for(int j = 0  ; j < _dim.y ; ++j)
            array[i][j]=val;
 

}

#include <iostream>
template <typename T>
void Array2DBorders<T>::freeMemory(){
using namespace std;

   if(array){
      
      for(int i = 0  ; i < internalDim.x ; ++i){
         delete [] array[i];
         array[i]=0;
      }

      delete [] array;
      array=0;

   }


}


template <typename T>
bool  Array2DBorders<T>::switchContainersQuick(Array2DBorders<T> & _switchedArray){
   ContainerType tmpPtr;
   ContainerType & switchedArrayPtr=_switchedArray.getContainer();

   tmpPtr=array;
   array=switchedArrayPtr;
   switchedArrayPtr=tmpPtr;
   return true;
}

//Adapter is necessary to keep current API of the Field3D . These fields are registered in Simulator and 
template <typename T>
class Array2DBordersField3DAdapter:public Field3DImpl<T>, public Array2DBorders<T>{
   public:
      Array2DBordersField3DAdapter() : Field3DImpl<T>(Dim3D(1,1,1),T()), Array2DBorders<T>() {};
      Array2DBordersField3DAdapter(Dim3D & _dim,T & _initVal):Field3DImpl<T>(Dim3D(1,1,1),T()), Array2DBorders<T>(){
         allocateMemory(_dim,_initVal);
         Array2DBorders<T>::internalDim=_dim;
      }
      virtual ~Array2DBordersField3DAdapter(){
//          if(array3DPtr)
//             delete array3DPtr;
//             array3DPtr=0;
      };
//       void setArray3DPtr(Array3D<T> * _ptr){ array3DPtr = _ptr;}
//       Array3DBorders<T> * getArray3DBordersPtr(){ return array3DPtr; }
//       typename Array3DBorders<T>::ContainerType &  getContainer(){ return getContainer(); }
      virtual void setDim(const Dim3D theDim) {
         T t;
         
         allocateMemory(theDim,t);
         Array2DBorders<T>::internalDim=theDim;

      }
      virtual Dim3D getDim() const {

         return Dim3D(Array2DBorders<T>::internalDim.x-2*Array2DBorders<T>::borderWidth,
                     Array2DBorders<T>::internalDim.y-2*Array2DBorders<T>::borderWidth,
                     1);
      };
      virtual bool isValid(const Point3D &pt) const {      
         return (0 <= pt.x && pt.x < Array2DBorders<T>::internalDim.x &&
	      0 <= pt.y && pt.y < Array2DBorders<T>::internalDim.y &&
	      0 == pt.z );
      }
      virtual void set(const Point3D &pt, const T value) {
         
//             unsigned int borderWidth=array3DPtr->getBorderWidth();
            Array2DBorders<T>::array
            [pt.x+Array2DBorders<T>::borderWidth]
            [pt.y+Array2DBorders<T>::borderWidth]
            =value;
         
      }
      virtual T get(const Point3D &pt) const {
//          unsigned int borderWidth=array3DPtr->getBorderWidth();
//          using namespace std;
//          cerr<<"this is element:"<<Array2DBorders<T>::array
//                   [pt.x+Array2DBorders<T>::borderWidth]
//                   [pt.y+Array2DBorders<T>::borderWidth]<<endl;
         return Array2DBorders<T>::array
                  [pt.x+Array2DBorders<T>::borderWidth]
                  [pt.y+Array2DBorders<T>::borderWidth];
      }

      virtual T getByIndex(long _offset) const {
         return T();
      }
      virtual void setByIndex(long _offset, const T _value) {
         
      }
   protected:
//       Array3DBorders<T> * array3DPtr;
//       Dim3D dim;
//       typename Array3DBorders<T>::ContainerType * containerPtr;
//       unsigned int borderWidth;
};

};



#endif
