###################################################################################################
#     The code was writen by Gilberto L Thomas for the 15th CC3D Wokshop. It is adapted from a previous one written 
#  and published on Biophys. J., 118 (2020) 2801 - https://doi.org/10.1016/j.bpj.2020.04.024.
####################################################################################################
#
# Defines the names of the parameters and persistent variables to be used here and in the steppables 
#
global cellRad, cellVol   # The cell linear size and its volume
global phiN, phiF, phiC  # Fractions of cell volume allocated to nucleus, lamellipodium, and cytoplasm respectively
global lambCHEM         # Strength by which the lamellipodia of cells respond to the FActin chemical field by chemotaxis
global Lx, Ly, Lz           # Parameters specifying the cell lattice size along x, y and z axes, respectively
global x1,y1,z1            # Initial center of mass position of the cell for x, y and z directions, respectively
global J0, T                 # Contact energy per voxel-voxel contact scale, and the Metropolis dynamics fluctuation amplitude, 
global tSim                 # Parameter specifying the duration of the simulation in MonteCarlo steps (mcs) 
global deltaT              # The interval between measurements by the analysis tools in mcs
global RANDOM_SEED  # Parameter specifying the seed for the pseudo-random number generator 
global g1Flag, g2Flag   # Parameter flags specifying what information to display as graphical output in CC3D Player
#
# Define the typical linear size of the cell
#
cellRad=10.0                  # The radius of the cell is in units of lattice spacing; 15., 20., 30.
cellVol=4.19*cellRad**3 # Define the volume of the cell in terms of its radius
#
# Defines relative cell compartment volumes
#
phiF=0.05                 # Fraction of cell volume allocated to FRONT Compartment, FRONT % cell volume; 0.10, 0.20, 0.30
phiN=0.15                # Fraction of cell volume allocated to NUCL compartment, NUCL % cell volume
phiC=1.- phiN - phiF  # Fraction of cell volume allocated to CYTO compartment, CYTO % cell volume
lambCHEM=-150.      # Strength \lambda_F-actin of cell's FRONT compartment response to F-actin "field" FActin; -125., -175., -200., -250.
#
# Defines grid size and CPM parameters
#                                    
Lx=int(8*cellRad) # Sets the size of the x axis of the cell lattice to be 8 times the cell radius by default
if cellRad>=20. and phiF >= 0.2:
   Lx=int(10*cellRad) 
if cellRad>=30. and phiF >= 0.2:
   Lx=int(12*cellRad) 
if cellRad>=40. and phiF >= 0.2:
   Lx=int(14*cellRad) 
Ly=Lx 
Lz=int(2*cellRad+2)
x1=int(Lx/2.);  y1=int(Ly/2.)    # Initial x-y cm position of the cell to be the x-y center of the cell lattice
T=100.                                 # Fluctuation amplitude
J0=20.                                 # Energy scale
tSim=100001                        # Defines the simualtion duration in mcs 
deltaT=50                            # Interval between measurements in mcs; =1 if velocity autocorrelation functions being calculated
RANDOM_SEED=68721           # Placeholder for the seed for the pseudo-random number generator from the ParameterScan file
#
# GRAPH plotting flags (yes/no): g1Flag - xy projection NUCL trajectory; g2Flag - others
#
g1Flag='yes' # Turn on display of graphs of cell movement properties
g2Flag='yes' # yes(no) - Turn on(off) display of graphs of cell movement properties                         
#
#     CC3D's "configureSimulation" function defines the model structure to be implemented in the simulation 
# (Objects, Properties, Behaviors, Interactions, Dynamics, Initial Conditions and Boundary Conditions)
#
def configureSimulation():
   # 
   from cc3d.core.XMLUtils import ElementCC3D # Housekeeping functions to allow configuration--the same in all CC3D models
   #
   cc3d=ElementCC3D("CompuCell3D",{"Revision":"20190811","Version":"4.0.0"}) # Identifies the version of CC3D for which the code was written
   #
   # Defines Basic properties of CPM (GGH) simulation (see CC3D manual)
   #
   Potts=cc3d.ElementCC3D("Potts")                                # Defines the beginning of the block specifying GGH/CPM-related features
   Potts.ElementCC3D("Dimensions",{"x":Lx,"y":Ly,"z":Lz}) # Defines lattice size of the simulation
   Potts.ElementCC3D("Steps",{},tSim)                           # Defines simulation duration in mcs
   Potts.ElementCC3D("Temperature",{},T)                      # Defines fluctuation amplitude in the MonteCarlo acceptance function
   Potts.ElementCC3D("RandomSeed",{},RANDOM_SEED)   # Initial seed for computational pseudo-random number generator
   Potts.ElementCC3D("NeighborOrder",{},1)                    # Range over which voxel-copies can occur, here 1st nearest neighbor only
   Potts.ElementCC3D("Boundary_x",{},"Periodic")            # Defines boundary conditions of the cell lattice as periodic in x
   Potts.ElementCC3D("Boundary_y",{},"Periodic")            # Defines boundary conditions of the cell lattice as periodic in y
   Potts.ElementCC3D("DebugOutputFrequency",{},1000)  # Opens a diagnostic window for simulation output
   #
   #    Defines the names of the generalized cell types in the simulation and the subcellular compartment components
   # and their corresponding integer TypeIds
   #    Note that some CC3D functions refer to CellTypes by TypeIds rather than CellType names
   #
   cellType=cc3d.ElementCC3D("Plugin",{"Name":"CellType"})
   cellType.ElementCC3D("CellType",{"TypeId":"0","TypeName":"Medium"})
   cellType.ElementCC3D("CellType",{"Freeze":"","TypeId":"1","TypeName":"SUBS_A"})
   cellType.ElementCC3D("CellType",{"Freeze":"","TypeId":"2","TypeName":"SUBS_NA"}) 
   cellType.ElementCC3D("CellType",{"TypeId":"3","TypeName":"CYTO"}) 
   cellType.ElementCC3D("CellType",{"TypeId":"4","TypeName":"FRONT"}) 
   cellType.ElementCC3D("CellType",{"TypeId":"5","TypeName":"NUCL"}) 
   #
   #    Defines certain terms in the effective energy and housekeeping functions used in simulation calculations
   #
   volumeLocalFlex=cc3d.ElementCC3D("Plugin",{"Name":"VolumeLocalFlex"})
   centerOfMass=cc3d.ElementCC3D("Plugin",{"Name":"CenterOfMass"})
   neighborTracker=cc3d.ElementCC3D("Plugin",{"Name":"NeighborTracker"})
   boundaryPixelTracker=cc3d.ElementCC3D("Plugin",{"Name":"BoundaryPixelTracker"})
   boundaryPixelTracker.ElementCC3D("NeighborOrder",{},"1")
   #
   #    We need to specify all the cell-cell and compartment-compartment contact energies
   # 
   #    The "Contact" plugin loads the calculation of the cell-cell and compartment-compartment contact energies, 
   # when these compartments belong to DIFFERENT cells 
   #
   contact=cc3d.ElementCC3D("Plugin",{"Name":"Contact"})
   contact.ElementCC3D("NeighborOrder",{},"4") # Contact energies are calculated  over voxels up to 4th neighbor 
   #
   #    "Energy" classes specify contact energies between compartments in DIFFERENT cells (all relative to J)=0).
   #
   contact.ElementCC3D("Energy",{"Type1":"Medium","Type2":"Medium"}, 1.) 
   contact.ElementCC3D("Energy",{"Type1":"Medium","Type2":"SUBS_A"}, J0)
   contact.ElementCC3D("Energy",{"Type1":"Medium","Type2":"SUBS_NA"}, -J0)
   contact.ElementCC3D("Energy",{"Type1":"Medium","Type2":"CYTO"}, J0)
   contact.ElementCC3D("Energy",{"Type1":"Medium","Type2":"FRONT"}, 2.*J0/3)
   contact.ElementCC3D("Energy",{"Type1":"Medium","Type2":"NUCL"}, 5.*J0)
   #
   contact.ElementCC3D("Energy",{"Type1":"SUBS_A","Type2":"SUBS_A"}, 1.)
   contact.ElementCC3D("Energy",{"Type1":"SUBS_A","Type2":"SUBS_NA"}, 1.)
   contact.ElementCC3D("Energy",{"Type1":"SUBS_A","Type2":"CYTO"}, J0)
   contact.ElementCC3D("Energy",{"Type1":"SUBS_A","Type2":"FRONT"}, J0/3)
   contact.ElementCC3D("Energy",{"Type1":"SUBS_A","Type2":"NUCL"}, 5.*J0)
   #
   contact.ElementCC3D("Energy",{"Type1":"SUBS_NA","Type2":"SUBS_NA"}, 1.)
   contact.ElementCC3D("Energy",{"Type1":"SUBS_NA","Type2":"CYTO"}, J0)
   contact.ElementCC3D("Energy",{"Type1":"SUBS_NA","Type2":"FRONT"}, 2.*J0/3)
   contact.ElementCC3D("Energy",{"Type1":"SUBS_NA","Type2":"NUCL"}, 5.*J0)
   # 
   contact.ElementCC3D("Energy",{"Type1":"CYTO","Type2":"CYTO"}, 2.*J0)
   contact.ElementCC3D("Energy",{"Type1":"CYTO","Type2":"FRONT"}, 2.*J0)
   contact.ElementCC3D("Energy",{"Type1":"CYTO","Type2":"NUCL"}, 5.*J0)
   # 
   contact.ElementCC3D("Energy",{"Type1":"FRONT","Type2":"FRONT"}, 2.*J0)
   contact.ElementCC3D("Energy",{"Type1":"FRONT","Type2":"NUCL"}, 5.*J0)
   # 
   contact.ElementCC3D("Energy",{"Type1":"NUCL","Type2":"NUCL"}, 5.*J0)
   #
   #    The "ContactInternal" plugin loads the calculation of the compartment-compartment contact energies, 
   # when these compartments belong to the SAME cell
   #
   intContact=cc3d.ElementCC3D("Plugin",{"Name":"ContactInternal"})
   intContact.ElementCC3D("NeighborOrder",{},"4") 
   # 
   intContact.ElementCC3D("Energy",{"Type1":"CYTO","Type2":"CYTO"}, 0.)
   intContact.ElementCC3D("Energy",{"Type1":"CYTO","Type2":"FRONT"}, J0/2) 
   intContact.ElementCC3D("Energy",{"Type1":"CYTO","Type2":"NUCL"}, J0)
   # 
   intContact.ElementCC3D("Energy",{"Type1":"FRONT","Type2":"FRONT"}, 0.0)
   intContact.ElementCC3D("Energy",{"Type1":"FRONT","Type2":"NUCL"}, 2.*J0)
   # 
   intContact.ElementCC3D("Energy",{"Type1":"NUCL","Type2":"NUCL"}, 0.)
   #
   # Specification of diffusing FActin field object and its interactions with generalized cell objects 
   #
   DiffusionSolverFE =cc3d.ElementCC3D("Steppable",{"Type":"DiffusionSolverFE"}) # Loads the diffusion equation solver
   Field=DiffusionSolverFE.ElementCC3D("DiffusionField")                     # Creates a diffusing field 
   Data=Field.ElementCC3D("DiffusionData")                                       # Specifies the start of the field specification
   Data.ElementCC3D("FieldName",{},"FActin")                                   # Names the diffusing field to be FActin
   Data.ElementCC3D("GlobalDiffusionConstant",{},1e-04)                 # Specifies the diffusion constant, D_F,
   Data.ElementCC3D("GlobalDecayConstant",{},0.9)                        # Specifies the decay constant, in units of 1/mcs
   Secretion=Field.ElementCC3D("SecretionData")                              # Specifies from where the field will come from
   Secretion.ElementCC3D("SecretionOnContact",{"SecreteOnContactWith":"SUBS_A","Type":"FRONT"},0.9) 
                                        # This secretion represents the rate of formation of F-actin fibers in the cell's lamellipodium.
   #
   #    The "Chemotaxis" plugin specifies the effective energy of chemotaxis 
   # due to interaction with the chemical field FActin
   #  
   Chemotaxis =cc3d.ElementCC3D("Plugin",{"Name":"Chemotaxis"}) # loads the "Chemotaxis" plug-in
   Field=Chemotaxis.ElementCC3D("ChemicalField",{"Name":"FActin", "Source":"DiffusionSolverFE"})    
   Field.ElementCC3D("ChemotaxisByType",{"ChemotactTowards":"Medium","Type":"FRONT","Lambda":lambCHEM}) #  
   #
   #   The uniform initializer is used to create the initial conditions of the simulation
   #
   UniformInitializer=cc3d.ElementCC3D("Steppable",{"Type":"UniformInitializer"})    
   # Initial layout of cells in the form of rectangular slab
   RegionElmnt=UniformInitializer.ElementCC3D("Region")
   RegionElmnt.ElementCC3D("BoxMin",{"x":"0","y":"0","z":"0"})
   RegionElmnt.ElementCC3D("BoxMax",{"x":Lx,"y":Ly,"z":"1"})
   RegionElmnt.ElementCC3D("Width",{},1)
   RegionElmnt.ElementCC3D("Types",{},"SUBS_A")
   #
   RegionElmnt=UniformInitializer.ElementCC3D("Region")
   RegionElmnt.ElementCC3D("BoxMin",{"x":"0","y":"0","z":Lz-1})
   RegionElmnt.ElementCC3D("BoxMax",{"x":Lx,"y":Ly,"z":Lz})
   RegionElmnt.ElementCC3D("Width",{},1)
   RegionElmnt.ElementCC3D("Types",{},"SUBS_NA")
   #
   RegionElmnt=UniformInitializer.ElementCC3D("Region")
   RegionElmnt.ElementCC3D("BoxMin",{"x":x1-cellRad,"y":y1-cellRad,"z":1})
   RegionElmnt.ElementCC3D("BoxMax",{"x":x1+cellRad,"y":y1+cellRad,"z":cellRad*2})
   RegionElmnt.ElementCC3D("Width",{},cellRad*2)
   RegionElmnt.ElementCC3D("Types",{},"CYTO")
   #
   RegionElmnt=UniformInitializer.ElementCC3D("Region")
   RegionElmnt.ElementCC3D("BoxMin",{"x":x1-3,"y":y1-3,"z":cellRad-3})
   RegionElmnt.ElementCC3D("BoxMax",{"x":x1+3,"y":y1+3,"z":cellRad+3})
   RegionElmnt.ElementCC3D("Width",{},6)
   RegionElmnt.ElementCC3D("Types",{},"NUCL")
   #
   # x2=x1-2*cellRad
   # y2=y1-2*cellRad
   
   CompuCellSetup.setSimulationXMLDescription(cc3d) # Applies the model structure specified 
                                                    # in the definition of the function "configureSimulation"

from cc3d import CompuCellSetup

configureSimulation()
#
#     The current simulation has two steppables "Cell" and "Calc"
#
from .CellMig3D_Demo_2020_Steppables import Cell 
# This steppable creates the cell and updates its compartments
CompuCellSetup.register_steppable(steppable=Cell(frequency=1,_phiF=phiF,_phiN=phiN,_cellVol=cellVol))
#
from .CellMig3D_Demo_2020_Steppables import Calc 
# This steppable calculates and stores cell's data
CompuCellSetup.register_steppable(steppable=Calc(frequency=1,_phiF=phiF,_phiN=phiN,_lambCHEM=lambCHEM,
                                                             _cellRad=cellRad,_g1Flag=g1Flag,_g2Flag=g2Flag,_x1=x1,_y1=y1,_dT=deltaT)) 

CompuCellSetup.run() # Starts the simulation 